// Photography Booking Website - Version 4 (HTML/CSS/JS)

const styles = [
  { 
    id: 'portrait', 
    name: 'Портрет', 
    icon: 'icons/style-portrait.svg',
    description: 'Индивидуальные портретные съёмки',
    duration: '1-2 часа'
  },
  { 
    id: 'family', 
    name: 'Семейная', 
    icon: 'icons/style-family.svg',
    description: 'Семейные фотосессии',
    duration: '2-3 часа'
  },
  { 
    id: 'lovestory', 
    name: 'Love Story', 
    icon: 'icons/style-lovestory.svg',
    description: 'Романтические съёмки',
    duration: '2-4 часа'
  },
  { 
    id: 'brand', 
    name: 'Бренд', 
    icon: 'icons/style-brand.svg',
    description: 'Брендовые фотосессии',
    duration: '3-5 часов'
  }
]

const photographers = [
  { name: 'Анна Петрова', experience: '8 лет', rating: 5, specialty: 'Love Story, Портрет' },
  { name: 'Михаил Соколов', experience: '12 лет', rating: 5, specialty: 'Бренд, Портрет' },
  { name: 'Елена Волкова', experience: '6 лет', rating: 5, specialty: 'Семейная, Love Story' },
  { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Бренд, Портрет' }
]

const features = [
  { icon: 'icons/stat-rating.svg', title: 'Профессиональное оборудование', text: 'Топовое фотооборудование' },
  { icon: 'icons/stat-sessions.svg', title: 'Гибкий график', text: 'Подстраиваемся под вас' },
  { icon: 'icons/stat-photographer.svg', title: 'Опытные фотографы', text: 'Многолетний опыт работы' },
  { icon: 'icons/form-camera.svg', title: 'Быстрая обработка', text: 'Готовые фото за 3-5 дней' }
]

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadSessions()
  updateStats()
  
  setInterval(() => {
    loadSessions()
    updateStats()
  }, 500)
}

function loadSessions() {
  const saved = localStorage.getItem('photoSessions')
  let sessions = []
  
  if (saved) {
    try {
      sessions = JSON.parse(saved)
    } catch (e) {
      sessions = []
    }
  } else {
    sessions = [
      {
        id: 1,
        photographer: 'Анна Петрова',
        theme: 'Love Story',
        date: '2025-03-15',
        rating: 5
      },
      {
        id: 2,
        photographer: 'Михаил Соколов',
        theme: 'Портрет',
        date: '2025-03-10',
        rating: 4
      },
      {
        id: 3,
        photographer: 'Елена Волкова',
        theme: 'Семейная',
        date: '2025-03-05',
        rating: 5
      }
    ]
    localStorage.setItem('photoSessions', JSON.stringify(sessions))
  }
  
  renderSessions(sessions)
  return sessions
}

function renderSessions(sessions) {
  const sessionsVertical = document.getElementById('sessionsVertical')
  if (!sessionsVertical) return
  
  if (sessions.length === 0) {
    sessionsVertical.innerHTML = `
      <div class="empty-state">
        <img src="icons/form-camera.svg" alt="" class="empty-icon" width="56" height="56" />
        <h3 class="empty-title">Нет запланированных съёмок</h3>
        <p class="empty-text">Запишитесь на фотосессию</p>
      </div>
    `
  } else {
    sessionsVertical.innerHTML = sessions.map(session => {
      const stars = Array.from({ length: 5 }, (_, i) => 
        i < session.rating 
          ? `<img src="icons/stat-rating.svg" alt="star" width="16" height="16" style="filter: drop-shadow(0 0 4px rgba(167, 139, 250, 0.5));" />`
          : `<img src="icons/stat-rating.svg" alt="star" width="16" height="16" style="opacity: 0.3;" />`
      ).join('')
      
      return `
        <div class="session-item">
          <div class="session-item-left">
            <div class="session-item-icon">
              <img src="icons/form-camera.svg" alt="" width="24" height="24" />
            </div>
            <div class="session-item-content">
              <div class="session-item-header">
                <h3 class="session-item-theme">${session.theme}</h3>
                <div class="session-item-rating">
                  ${stars}
                </div>
              </div>
              <div class="session-item-details">
                <div class="session-item-detail">
                  <img src="icons/stat-photographer.svg" alt="" width="16" height="16" />
                  <span>${session.photographer}</span>
                </div>
                <div class="session-item-detail">
                  <img src="icons/form-calendar.svg" alt="" width="16" height="16" />
                  <span>
                    ${new Date(session.date).toLocaleDateString('ru-RU', { 
                      day: 'numeric', 
                      month: 'long', 
                      year: 'numeric' 
                    })}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const sessions = loadSessions()
  const stats = [
    { 
      value: sessions.length, 
      label: 'Съёмок'
    },
    { 
      value: new Set(sessions.map(s => s.photographer)).size, 
      label: 'Фотографов'
    },
    { 
      value: sessions.length > 0 
        ? (sessions.reduce((sum, s) => sum + s.rating, 0) / sessions.length).toFixed(1)
        : 0, 
      label: 'Рейтинг'
    }
  ]
  
  const statsMinimal = document.getElementById('statsMinimal')
  if (statsMinimal) {
    statsMinimal.innerHTML = stats.map(stat => `
      <div class="stat-minimal-item">
        <div class="stat-minimal-value">${stat.value}</div>
        <div class="stat-minimal-label">${stat.label}</div>
      </div>
    `).join('')
  }
}

function initBookPage() {
  const bookingForm = document.getElementById('bookingForm')
  const stylesList = document.getElementById('stylesList')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const photographersList = document.getElementById('photographersList')
  const featuresGrid = document.getElementById('featuresGrid')

  let selectedStyle = null

  if (stylesList) {
    stylesList.innerHTML = styles.map(style => `
      <button type="button" class="style-item" data-id="${style.id}">
        <div class="style-item-icon">
          <img src="${style.icon}" alt="${style.name}" width="24" height="24" />
        </div>
        <div class="style-item-content">
          <h3 class="style-item-name">${style.name}</h3>
          <p class="style-item-desc">${style.description}</p>
          <span class="style-item-duration">${style.duration}</span>
        </div>
      </button>
    `).join('')

    stylesList.querySelectorAll('.style-item').forEach(button => {
      button.addEventListener('click', function() {
        stylesList.querySelectorAll('.style-item').forEach(btn => {
          btn.classList.remove('selected')
          const check = btn.querySelector('.style-item-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        selectedStyle = styles.find(s => s.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'style-item-check'
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="18" height="18" />`
        this.appendChild(check)
      })
    })
  }

  if (photographersList) {
    photographersList.innerHTML = photographers.map(photographer => {
      const stars = Array.from({ length: 5 }, () => 
        `<img src="icons/stat-rating.svg" alt="star" width="14" height="14" style="filter: drop-shadow(0 0 4px rgba(167, 139, 250, 0.5));" />`
      ).join('')
      
      return `
        <div class="photographer-item">
          <div class="photographer-item-left">
            <div class="photographer-item-icon">
              <img src="icons/stat-photographer.svg" alt="" width="24" height="24" />
            </div>
            <div class="photographer-item-content">
              <h3 class="photographer-item-name">${photographer.name}</h3>
              <div class="photographer-item-details">
                <span class="photographer-item-exp">${photographer.experience} опыта</span>
                <div class="photographer-item-rating">
                  ${stars}
                </div>
              </div>
              <p class="photographer-item-specialty">${photographer.specialty}</p>
            </div>
          </div>
        </div>
      `
    }).join('')
  }

  if (featuresGrid) {
    featuresGrid.innerHTML = features.map(feature => `
      <div class="feature-item">
        <div class="feature-item-icon">
          <img src="${feature.icon}" alt="" width="24" height="24" />
        </div>
        <div class="feature-item-content">
          <h3 class="feature-item-title">${feature.title}</h3>
          <p class="feature-item-text">${feature.text}</p>
        </div>
      </div>
    `).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedStyle && dateInput.value) {
        const savedSessions = localStorage.getItem('photoSessions')
        const sessions = savedSessions ? JSON.parse(savedSessions) : []
        
        const randomPhotographer = photographers[Math.floor(Math.random() * photographers.length)]
        const newSession = {
          id: Date.now(),
          photographer: randomPhotographer.name,
          theme: selectedStyle.name,
          date: dateInput.value,
          rating: 0
        }

        sessions.push(newSession)
        localStorage.setItem('photoSessions', JSON.stringify(sessions))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedStyle = null
          dateInput.value = ''
          stylesList.querySelectorAll('.style-item').forEach(btn => {
            btn.classList.remove('selected')
            const check = btn.querySelector('.style-item-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  slides[currentSlide].classList.remove('active')
  dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

